import asyncio
import socket
import os
import time

from enum import IntEnum

from pylog.pylogger import PyLogger

from py_pli.pylib import VUnits

from py_pli.pyexception import UrpcFirmwareException

from urpc_enum.measurementparameter import MeasurementParameter

from urpc_enum.error import NodeErrorCode

from fleming.common.firmware_util import *

from virtualunits.vu_measurement_unit import VUMeasurementUnit
from virtualunits.meas_seq_generator import meas_seq_generator
from virtualunits.meas_seq_generator import TriggerSignal
from virtualunits.meas_seq_generator import OutputSignal
from virtualunits.meas_seq_generator import MeasurementChannel
from virtualunits.meas_seq_generator import IntegratorMode
from virtualunits.meas_seq_generator import AnalogControlMode


pmt_channel = {
    'pmt1' : MeasurementChannel.PMT1,
    'pmt2' : MeasurementChannel.PMT2,
}

pmt_dl = {
    'pmt1' : MeasurementParameter.PMT1DiscriminatorLevel,
    'pmt2' : MeasurementParameter.PMT2DiscriminatorLevel,
}

pmt_hv = {
    'pmt1' : MeasurementParameter.PMT1HighVoltageSetting,
    'pmt2' : MeasurementParameter.PMT2HighVoltageSetting,
}

pmt_hv_enable = {
    'pmt1' : MeasurementParameter.PMT1HighVoltageEnable,
    'pmt2' : MeasurementParameter.PMT2HighVoltageEnable,
}

pmt_hv_gate = {
    'pmt1' : OutputSignal.HVGatePMT1,
    'pmt2' : OutputSignal.HVGatePMT2,
}

pmt_input_gate = {
    'pmt1' : OutputSignal.InputGatePMT1,
    'pmt2' : OutputSignal.InputGatePMT2,
}

pmt_sample = {
    'pmt1' : TriggerSignal.SamplePMT1,
    'pmt2' : TriggerSignal.SamplePMT2,
}

pmt_conversion = {
    'pmt1' : EEFDigitalInput.PMT1CONVERSION,
    'pmt2' : EEFDigitalInput.PMT2CONVERSION,
}

integrator_mode = {
    'reset'     : IntegratorMode.full_reset,
    'low_reset' : IntegratorMode.low_range_reset,
    'auto'      : IntegratorMode.integrate_autorange,
    'fixed'     : IntegratorMode.integrate_with_fixed_range,
    'low'       : IntegratorMode.integrate_in_low_range,
    'high'      : IntegratorMode.integrate_in_high_range,
}


async def pmt_set_dl(pmt, dl):
    if (pmt not in pmt_channel):
        raise ValueError(f"pmt must be {pmt_channel}")

    delay = 0.1
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await meas_unit.endpoint.SetParameter(pmt_dl[pmt], dl, timeout=1)
    await asyncio.sleep(delay)


async def pmt_set_hv(pmt, hv):
    if (pmt not in pmt_channel):
        raise ValueError(f"pmt must be {pmt_channel}")

    delay = 0.1
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await meas_unit.endpoint.SetParameter(pmt_hv[pmt], hv, timeout=1)
    await asyncio.sleep(delay)
    

async def pmt_set_hv_enable(pmt, enable):
    if (pmt not in pmt_channel):
        raise ValueError(f"pmt must be {pmt_channel}")

    delay = 0.1
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    await meas_unit.endpoint.SetParameter(pmt_hv_enable[pmt], enable, timeout=1)
    await asyncio.sleep(delay)


async def pmt_set_hv_gate(pmt, enable):
    if (pmt not in pmt_channel):
        raise ValueError(f"pmt must be {pmt_channel}")

    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'pmt_set_hv_gate'
    seq_gen = meas_seq_generator()
    if (enable == 1):
        seq_gen.SetSignals(pmt_hv_gate[pmt])
    else:
        seq_gen.ResetSignals(pmt_hv_gate[pmt])
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)


async def pmt_set_input_gate(pmt, enable):
    if (pmt not in pmt_channel):
        raise ValueError(f"pmt must be {pmt_channel}")

    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'pmt_set_input_gate'
    seq_gen = meas_seq_generator()
    if (enable == 1):
        seq_gen.SetSignals(pmt_input_gate[pmt])
    else:
        seq_gen.ResetSignals(pmt_input_gate[pmt])
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)


async def pmt_set_integrator_mode(pmt, mode):
    if (pmt not in pmt_channel):
        raise ValueError(f"pmt must be {pmt_channel}")
    if (mode not in integrator_mode):
        raise ValueError(f"mode must be {integrator_mode}")

    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'pmt_set_integrator_mode'
    seq_gen = meas_seq_generator()
    seq_gen.SetIntegratorMode(**{pmt: integrator_mode[mode]})
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)


async def pmt_get_analog_result(pmt):
    if (pmt not in pmt_channel):
        raise ValueError(f"pmt must be {pmt_channel}")

    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'pmt_get_analog_result'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # analog_high_result
    seq_gen.SetTriggerOutput(pmt_sample[pmt])
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=1)
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    meas_unit.resultAddresses[op_id] = range(0, 2)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    return results


async def pmt_set_conversion(pmt, duration_ms=1000):
    if (pmt not in pmt_sample):
        raise ValueError(f"pmt must be {pmt_sample}")

    trigger_delay = 100                             # 1 us
    trigger_loop = round(100000 / trigger_delay)    # 1 ms / 1 us
    duration_ms = round(duration_ms)

    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'pmt_set_conversion'
    seq_gen = meas_seq_generator()
    seq_gen.Loop(duration_ms)
    seq_gen.Loop(trigger_loop)
    seq_gen.TimerWaitAndRestart(trigger_delay)
    seq_gen.SetTriggerOutput(pmt_sample[pmt])
    seq_gen.LoopEnd()
    seq_gen.LoopEnd()
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)


async def pmt_get_conversion(pmt):
    if (pmt not in pmt_conversion):
        raise ValueError(f"pmt must be {pmt_conversion}")

    eef = get_node_endpoint('eef')

    return (await eef.GetDigitalInput(pmt_conversion[pmt]))[0]


async def pmt_measure_v7(pmt, window_us, fixed_range_us=20):
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    full_reset_delay = 40000000     # 400 us
    conversion_delay = 1200         #  12 us
    switch_delay = 25               # 250 ns
    gating_delay = 100              #   1 us

    if (pmt not in pmt_channel):
        raise ValueError(f"pmt must be {pmt_channel}")
    if (fixed_range_us < 0) or (fixed_range_us > 671088):
        raise ValueError(f"fixed_range_us must be in the range [0, 671088] us")
        
    window_us = round(window_us)
    window_corse, window_fine = divmod(window_us, 65536)
    pre_cnt_window = 100    # 1 µs
    fixed_range = round(fixed_range_us * 100)

    op_id = 'pmt_measure_v7'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # count_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # analog_high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # analog_low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # analog_high_offset
    # Reset the offsets
    seq_gen.SetAnalogControl(**{pmt: AnalogControlMode.full_offset_reset})
    # Enable full reset and disable the analog input
    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(**{pmt: IntegratorMode.full_reset})
    seq_gen.ResetSignals(pmt_input_gate[pmt])
    # Measure the high range offset
    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(pmt_sample[pmt])
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=4)
    seq_gen.SetAnalogControl(**{pmt: AnalogControlMode.read_offset})
    # Start the integrator in low range (250 ns before the low range offset is measured -> TBD)
    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(**{pmt: IntegratorMode.low_range_reset})
    # Measure the low range offset, then switch to auto range and enable the analog input
    seq_gen.TimerWaitAndRestart(gating_delay)
    seq_gen.SetTriggerOutput(pmt_sample[pmt])
    seq_gen.SetIntegratorMode(**{pmt: IntegratorMode.integrate_autorange})
    seq_gen.SetSignals(pmt_input_gate[pmt])
    # Start counting
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(pmt_channel[pmt], cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if (window_corse > 0):
        seq_gen.Loop(window_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(pmt_channel[pmt], cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if (window_fine > 0):
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(pmt_channel[pmt], cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()
    # Disable the analog input
    seq_gen.ResetSignals(pmt_input_gate[pmt])
    # Switch to fixed range
    seq_gen.TimerWaitAndRestart(fixed_range)
    seq_gen.SetIntegratorMode(**{pmt: IntegratorMode.integrate_with_fixed_range})
    # Save the low range offset after the measurement window, since we must not block the pre-counter loop by waiting for the analog result
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=4)
    seq_gen.SetAnalogControl(**{pmt: AnalogControlMode.read_offset})
    # Measure the analog result and get the counting result
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(pmt_sample[pmt])
    seq_gen.GetPulseCounterResult(pmt_channel[pmt], relative=False, resetCounter=True, cumulative=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=2)
    # Enable full reset
    seq_gen.SetIntegratorMode(**{pmt: IntegratorMode.full_reset})
    # Reset the offsets (only needed in case a sequence without offset correction is called after this)
    seq_gen.SetAnalogControl(**{pmt: AnalogControlMode.full_offset_reset})
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    meas_unit.resultAddresses[op_id] = range(0, 5)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    PyLogger.logger.info(f"Count: {results[0]:10d} ; Low: {results[1]:5d} ; High: {results[2]:5d} ; (Offset: {results[3]:5d} ; {results[4]:5d})")

    return results


async def pmt_analog_window_scan(pmt, start_us=1, stop_us=100, step_us=1, average=16, hv=0.6, dl=0.25):
    analog_high_range_scale = 10.4

    PyLogger.logger.info(f"Start EEF firmware")
    await start_firmware('eef')

    await pmt_set_dl(pmt, dl)
    await pmt_set_hv(pmt, hv)
    await pmt_set_hv_enable(pmt, enable=True)
    await pmt_set_hv_gate(pmt, enable=True)

    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    with open(f"pmt_analog_window_scan_{timestamp}.csv", 'w') as file:
        file.write(f"pmt_analog_window_scan(pmt={pmt}, start_us={start_us}, stop_us={stop_us}, step_us={step_us}, average={average}, hv={hv}, dl={dl})\n")
        file.write(f"window_us ; analog_signal ; analog_low  ; analog_high\n")

        window_us_range = [window_us / 1e6 for window_us in range(round(start_us * 1e6), round(stop_us * 1e6 + 1), round(step_us * 1e6))]
        for window_us in window_us_range:
            results_sum = [0]*5
            for i in range(average):
                results = await pmt_measure_v7(pmt, window_us, fixed_range_us=20)
                results_sum = [a + b for a, b in zip(results_sum, results)]
            
            results = [result / average for result in results_sum]
            analog_signal = results[1] + results[2] * analog_high_range_scale
            file.write(f"{window_us} ; {analog_signal} ; {results[1]} ; {results[2]}\n")

    await pmt_set_hv_gate(pmt, enable=False)
    await pmt_set_hv_enable(pmt, enable=False)

    return f"pmt_analog_window_scan() done"


async def pmt_measure_input_gate(pmt, window_us=20, switch_delay=1):
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    full_reset_delay = 40000000     # 400 us
    low_reset_delay  = 25           # 250 ns
    auto_range_delay = 25           # 250 ns
    conversion_delay = 1200         #  12 us

    if (pmt not in pmt_channel):
        raise ValueError(f"pmt must be {pmt_channel}")
    if (switch_delay < 1) or (switch_delay > 67108864):
        raise ValueError(f"switch_delay must be in the range [1, 67108864]")
        
    window = round(window_us * 100)
    if (window <= (conversion_delay + switch_delay)) or (window > 67108864):
        raise ValueError(f"window_us is out of range")

    op_id = 'pmt_input_gate_test'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # Input Gate Low (analog_low)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # Input Gate Low (analog_high)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # Input Gate High (analog_low)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # Input Gate High (analog_high)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)   # Input Gate High (analog_low)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)   # Input Gate High (analog_high)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)   # Input Gate Low (analog_low)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)   # Input Gate Low (analog_high)
    # Enable full reset and disable the analog input
    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(**{pmt: IntegratorMode.full_reset})
    seq_gen.ResetSignals(pmt_input_gate[pmt])
    # Switch to low reset
    seq_gen.TimerWaitAndRestart(low_reset_delay)
    seq_gen.SetIntegratorMode(**{pmt: IntegratorMode.low_range_reset})
    # Switch to auto range
    seq_gen.TimerWaitAndRestart(auto_range_delay)
    seq_gen.SetIntegratorMode(**{pmt: IntegratorMode.integrate_autorange})
    # First measurement
    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(pmt_sample[pmt])
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=1)
    # Set input gate high (start of the measurement window)
    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetSignals(pmt_input_gate[pmt])
    # Second measurement
    seq_gen.TimerWaitAndRestart(window - switch_delay)
    seq_gen.SetTriggerOutput(pmt_sample[pmt])
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=3)
    # Third measurement
    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(pmt_sample[pmt])
    # Set input gate low (end of measurement window)
    seq_gen.ResetSignals(pmt_input_gate[pmt])
    # Swich to fixed range
    seq_gen.SetIntegratorMode(**{pmt: IntegratorMode.integrate_with_fixed_range})
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=5)
    # Fourth measurement
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(pmt_sample[pmt])
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=6)
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=7)
    # Enable full reset
    seq_gen.SetIntegratorMode(**{pmt: IntegratorMode.full_reset})
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    meas_unit.resultAddresses[op_id] = range(0, 8)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    PyLogger.logger.info(f"t1: {results[0]:5d} ; {results[1]:5d}")
    PyLogger.logger.info(f"t2: {results[2]:5d} ; {results[3]:5d}")
    PyLogger.logger.info(f"t3: {results[4]:5d} ; {results[5]:5d}")
    PyLogger.logger.info(f"t4: {results[6]:5d} ; {results[7]:5d}")

    return results


async def pmt_input_gate_test(pmt, iterations, window_us=20, switch_delay=1):
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    with open(f"pmt_input_gate_test_{timestamp}.csv", 'w') as file:
        file.write(f"pmt_input_gate_test(pmt={pmt}, window_us={window_us}, switch_delay={switch_delay})\n")
        file.write(f"t1_low  ; t1_high ; t2_low  ; t2_high ; t3_low  ; t3_high ; t4_low  ; t4_high\n")
        for i in range(iterations):
            results = await pmt_measure_input_gate(pmt, window_us, switch_delay)
            file.write(f"{results[0]:7d} ; {results[1]:7d} ; {results[2]:7d} ; {results[3]:7d} ; {results[4]:7d} ; {results[5]:7d} ; {results[6]:7d} ; {results[7]:7d}\n")

    return f"pmt_input_gate_test() done"


async def pmt_analog_leakage_test(pmt, duration_s):
    if (pmt not in pmt_channel):
        raise ValueError(f"pmt must be {pmt_channel}")
    
    SAMPLE_DELAY_S = 1.0

    PyLogger.logger.info(f"Start EEF firmware")
    await start_firmware('eef')
    
    comment = input(f"Enter a comment ('X' to cancel): ")
    if comment == 'X' or comment == 'x':
        return f"pmt_analog_leakage_test() canceled"
    
    await pmt_set_integrator_mode(pmt, 'reset')
    await asyncio.sleep(0.1)
    await pmt_set_integrator_mode(pmt, 'low_reset')
    await pmt_set_integrator_mode(pmt, 'auto')

    await pmt_set_input_gate(pmt, True)

    instrument = socket.gethostname()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    with open(f"pmt_analog_leakage_test_{timestamp}.csv", 'w') as file:
        file.write(f"pmt_analog_leakage_test_(pmt={pmt}, duration_s={duration_s:.1f}) on {instrument} started at {timestamp}\n")
        file.write(f"comment: {comment}\n")
        file.write(f"time [s] ; analog_low  ; analog_high\n")
        
        start = time.perf_counter()
        timestamp = 0.0
        while (timestamp < duration_s):
            results = await asyncio.gather(
                pmt_get_analog_result(pmt),
                asyncio.sleep(SAMPLE_DELAY_S)
            )
            analog_low  = results[0][0]
            analog_high = results[0][1]
            file.write(f"{timestamp:8.1f} ; {analog_low:11d} ; {analog_high:11d}\n")
            PyLogger.logger.info(f"time: {timestamp:8.1f} ; analog_low: {analog_low:5d} ; analog_high: {analog_high:11d}")
            timestamp = time.perf_counter() - start

    await pmt_set_integrator_mode(pmt, 'reset')
    await pmt_set_input_gate(pmt, False)

    return f"pmt_analog_leakage_test() done"


async def pmt_measure_analog(pmt, window_us=50, trigger_delay_us=0):
    if (pmt not in pmt_channel):
        raise ValueError(f"pmt must be {pmt_channel}")
    if (window_us < 0) or (window_us > 671088):
        raise ValueError(f"window_us must be in the range [0, 671088] us")
    if (trigger_delay_us < 0) or (trigger_delay_us > 671088):
        raise ValueError(f"trigger_delay_us must be in the range [0, 671088] us")
    if (trigger_delay_us >= window_us):
        raise ValueError(f"trigger_delay_us must be less than window_us")

    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    full_reset_delay = 40000000     # 400 us
    conversion_delay = 1200         #  12 us
    switch_delay = 25               # 250 ns

    if (pmt not in pmt_channel):
        raise ValueError(f"pmt must be {pmt_channel}")

    window = round(window_us * 100)
    trigger_delay = round(trigger_delay_us * 100)
    fixed_range = 2000  # 20 µs

    op_id = 'pmt_measure_analog'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # analog_low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # analog_high_result
    # Reset the offsets
    seq_gen.SetAnalogControl(**{pmt: AnalogControlMode.full_offset_reset})
    # Enable full reset and disable the analog input
    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(**{pmt: IntegratorMode.full_reset})
    seq_gen.ResetSignals(pmt_input_gate[pmt])
    # Measure the high range offset
    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(pmt_sample[pmt])
    # Start the integrator in low range (250 ns before the low range offset is measured -> TBD)
    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(**{pmt: IntegratorMode.low_range_reset})
    seq_gen.SetAnalogControl(**{pmt: AnalogControlMode.read_offset})
    # Measure the low range offset, then switch to auto range and enable the analog input
    seq_gen.TimerWaitAndRestart(trigger_delay)
    seq_gen.SetTriggerOutput(pmt_sample[pmt])
    seq_gen.SetIntegratorMode(**{pmt: IntegratorMode.integrate_autorange})
    seq_gen.SetSignals(pmt_input_gate[pmt])
    # Send the trigger pulse
    seq_gen.TimerWaitAndRestart(window - trigger_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.TRF)
    # Disable the analog input, then switch to fixed range
    seq_gen.TimerWaitAndRestart(fixed_range)
    seq_gen.ResetSignals(pmt_input_gate[pmt])
    seq_gen.SetIntegratorMode(**{pmt: IntegratorMode.integrate_with_fixed_range})
    # Save the low range offset after the measurement window, since we must not block the pre-counter loop by waiting for the analog result
    seq_gen.SetAnalogControl(**{pmt: AnalogControlMode.read_offset})
    # Measure the analog result
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(pmt_sample[pmt])
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=1)
    # Enable full reset
    seq_gen.SetIntegratorMode(**{pmt: IntegratorMode.full_reset})
    # Reset the offsets (only needed in case a sequence without offset correction is called after this)
    seq_gen.SetAnalogControl(**{pmt: AnalogControlMode.full_offset_reset})
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    meas_unit.resultAddresses[op_id] = range(0, 2)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    PyLogger.logger.info(f"analog_low: {results[0]:5d} ; analog_high: {results[1]:5d}")

    return results


async def pmt_analog_pulse_test(pmt, iterations=10, window_us=50, trigger_delay_us=0):

    SAMPLE_DELAY_S = 0.1

    PyLogger.logger.info(f"Start EEF firmware")
    await start_firmware('eef')
    
    comment = input(f"Enter a comment ('X' to cancel): ")
    if comment == 'X' or comment == 'x':
        return f"pmt_analog_pulse_test() canceled"

    instrument = socket.gethostname()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    with open(f"pmt_analog_pulse_test_{timestamp}.csv", 'w') as file:
        file.write(f"pmt_analog_pulse_test(pmt={pmt}, iterations={iterations}, window_us={window_us:.1f}, trigger_delay_us={trigger_delay_us:.1f}) on {instrument} started at {timestamp}\n")
        file.write(f"comment: {comment}\n")
        file.write(f"iteration ; analog_low  ; analog_high\n")
        
        for i in range(iterations):
            analog_low, analog_high = await pmt_measure_analog(pmt, window_us, trigger_delay_us)
            file.write(f"{(i+1):9d} ; {analog_low:11d} ; {analog_high:11d}\n")
            await asyncio.sleep(SAMPLE_DELAY_S)

    return f"pmt_analog_pulse_test() done"


async def pmt_noise_test(pmt, sample_rate=83333, sample_count=4096):
    if (pmt not in pmt_channel):
        raise ValueError(f"pmt must be {pmt_channel}")
    if (sample_rate < 2) or (sample_rate > 83333):
        raise ValueError(f"sample_rate must be in the range [2, 83333] Hz")
    if (sample_count < 1) or (sample_count > 4096):
        raise ValueError(f"sample_count must be in the range [1, 4096]")

    await send_gc_msg(f"Start EEF firmware")
    await start_firmware('eef')

    instrument = socket.gethostname()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    report_dir = f"{os.path.dirname(__file__)}/pmt_test_results"
    os.makedirs(report_dir, exist_ok=True)
    with open(f"{report_dir}/pmt_noise_test__{instrument}_{timestamp}.csv", 'w') as file:
        await write_gc_msg(file, f"pmt_noise_test(pmt={pmt}, sample_rate={sample_rate}, sample_count={sample_count}) on {instrument} started at {timestamp}")
        await write_gc_msg(file, f"time [ms] ; full_reset ; low_range  ; high_range")

        full_reset_noise = await pmt_noise_measurement(pmt=pmt, mode='reset', sample_rate=sample_rate, sample_count=sample_count)
        low_range_noise  = await pmt_noise_measurement(pmt=pmt, mode='low',   sample_rate=sample_rate, sample_count=sample_count)
        high_range_noise = await pmt_noise_measurement(pmt=pmt, mode='high',  sample_rate=sample_rate, sample_count=sample_count)

        time_ms = 0.0
        for i in range(sample_count):
            await write_gc_msg(file, f"{time_ms:9.3f} ; {full_reset_noise[i]:10d} ; {low_range_noise[i]:10d} ; {high_range_noise[i]:10d}")
            time_ms += 1000.0 / sample_rate

    return f"pmt_noise_test() done"


async def pmt_noise_measurement(pmt, mode, sample_rate=83333, sample_count=1000):
    if (pmt not in pmt_channel):
        raise ValueError(f"pmt must be {pmt_channel}")
    if (mode not in integrator_mode):
        raise ValueError(f"mode must be {integrator_mode}")
    if (sample_rate < 2) or (sample_rate > 83333):
        raise ValueError(f"sample_rate must be in the range [2, 83333] Hz")
    if (sample_count < 1) or (sample_count > 4096):
        raise ValueError(f"sample_count must be in the range [1, 4096]")
        
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    full_reset_delay = 40000    # 400 us
    sample_delay = round(100000000 / sample_rate)   # 10 ns resolution

    op_id = 'pmt_noise_measurement'
    seq_gen = meas_seq_generator()
    # Clear the result buffer
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    seq_gen.Loop(sample_count)
    seq_gen.ClearResultBuffer(relative=True, dword=False, addrReg=0, addr=0)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd()
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    # Enable full reset
    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(**{pmt: IntegratorMode.full_reset})
    # Sample the signal
    seq_gen.TimerWaitAndRestart(sample_delay)
    seq_gen.SetIntegratorMode(**{pmt: integrator_mode[mode]})
    seq_gen.Loop(sample_count)
    seq_gen.TimerWaitAndRestart(sample_delay)
    seq_gen.SetTriggerOutput(pmt_sample[pmt])
    seq_gen.GetAnalogResult(pmt_channel[pmt], isRelativeAddr=True, ignoreRange=True, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd()
    # End of measurement
    seq_gen.SetIntegratorMode(**{pmt: IntegratorMode.full_reset})
    seq_gen.Stop(0)

    meas_unit.ClearOperations()
    meas_unit.resultAddresses[op_id] = range(0, sample_count)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    return results


async def pmt_connection_test(pmt):

    await send_gc_msg(f"Start EEF firmware")
    await start_firmware('eef')
    
    try:
        if (await pmt_get_conversion(pmt)) != 0:
            return f"pmt_connection_test() sanity check failed"
    except UrpcFirmwareException as ex:
        if ex.errorCode == NodeErrorCode.DeviceNotReady:
            return f"{pmt} not connected"
        else:
            raise

    await pmt_set_conversion(pmt, 1000)
    if (await pmt_get_conversion(pmt)) != 1:
        return f"{pmt} cable orientation incorrect"

    await asyncio.sleep(1.1)
    if (await pmt_get_conversion(pmt)) != 0:
        return f"pmt_connection_test() sanity check failed"

    return f"{pmt} connected"

